#ifndef V3D_TOOLS_MATHS_H_INCLUDED
#define V3D_TOOLS_MATHS_H_INCLUDED

#include <cmath>


namespace V3D {

	extern const double k_dPiDiv2;
	extern const double k_dPi;
	extern const double k_d2Pi;

	extern const float k_fMinFloat;
	extern const float k_fMaxFloat;
	extern const float k_fPiDiv2;
	extern const float k_fPi;
	extern const float k_f2Pi;
	
	extern const double k_dSqrt2;
	extern const double k_dSqrt3;
	extern const double k_dSqrt3Div2;

	extern const float k_fSqrt2;
	extern const float k_fSqrt3;
	extern const float k_fSqrt3Div2;

	template <class T> inline const T& Min(const T& t1, const T& t2) { return ( t1 < t2) ? t1 : t2;}
	template <class T> inline const T& Max(const T& t1, const T& t2) { return ( t1 > t2) ? t1 : t2;}
	
	inline int32 Max( int32 n1, int32 n2) 
	{
		int32 nDiffMask = (n1 - n2) >> 31; // 0xFFFFFFFF si n1 < n2, 0 sinon
		return (n1 & ~nDiffMask) | (n2 & nDiffMask);
	}

	inline int32 Min( int32 n1, int32 n2) 
	{
		int32 nDiffMask = (n1 - n2) >> 31; // 0xFFFFFFFF si n1 < n2, 0 sinon
		return (n1 & nDiffMask) | (n2 & ~nDiffMask);
	}		
	
	template <class T> inline const T& Clamp(const T& t, const T& tmin, const T& tmax) 
		{ return ( Max( tmin, Min( tmax, t ) ) ); }

	
	template <class T> inline T Sqr(const T& t) { return (t*t);}
	
	template <class T> inline T Abs( const T t) { return (t>0) ? t : -t;}
	template <> inline float  Abs( float t) { return float(fabs(t));}
	template <> inline double Abs( double t) { return fabs(t);}
	template <> inline int32  Abs( int32 n) { int32 nMask = n >> 31; return (n^nMask)-nMask;}
	template <> inline int16  Abs( int16 n) { int16 nMask = n >> 15; return (n^nMask)-nMask;}
	template <> inline int8  Abs( int8 n) { int8 nMask = n >> 7; return (n^nMask)-nMask;}


	extern int32 Floor( const double f);
	extern int32 Ceil( const double f);

	extern int32 FloorFast( const double f);
	extern int32 CeilFast( const double f);

/*
	inline float Floor( const float f)
		{ return (float) floor(f);}

	inline float Ceil( const float f)
		{ return (float) ceil(f);}
*/
	inline float Sqrt( const float f)
		{ return sqrtf(f);}

	inline double Sqrt( const double f)
		{ return sqrt(f);}

	inline float Cos( const float f)
		{ return float(cos(f));}

	inline float Sin( const float f)
		{ return float(sin(f));}

	inline float Tan( const float f)
		{ return float(tan(f));}

	inline float Acos( const float f)
		{ return float(acos(f));}

	inline float Atan2( const float f1, const float f2)
		{ return float(atan2(f1, f2));}

	inline float Asin( const float f)
		{ return float(asin(f));}

		
	inline float CubicRoot( const float f)
	{
		if( f > 0)
			return (float) pow(f, 1.f / 3.f);
		else
			return (float) - pow(-f, 1.f / 3.f); // evite les erreurs (la puissance 1/3 de valeurs negatives est instable)
	}

	inline int32 FastFloatGetSign( float fFloat)
	{
		uint32 nFloat = * ( (uint32*) &fFloat);
		return nFloat >> (8 * sizeof( float ) - 1);
	}

	inline int32 FastFloatGetSign( double dDbl)
	{
		uint64 nDbl = * ( (uint64*) &dDbl);
		return int32(nDbl >> ( 8 * sizeof( double ) - 1) );
	}


	inline bool AreFloatsDiffSigns( float fFloat1, float fFloat2)
	{
		uint32 nFloat1 = (* ( (uint32*) &fFloat1));
		uint32 nFloat2 = (* ( (uint32*) &fFloat2));
		return ((nFloat1 ^ nFloat2) & 0x80000000u) != 0;
	}

	// Nombre aleatoire entre 0 et 1
	inline float Random()
	{ return float( rand() ) / RAND_MAX; }



} // namespaces




#endif		// #ifndef V3D_TOOLS_MATHS_H_INCLUDED
